﻿#pragma once
#include <QOpenGLFunctions_3_3_Core>
#include <QVector3D>
#include <vector>

// 颜色常量定义
const float PLANE_COLOR_R = 0.839f;
const float PLANE_COLOR_G = 0.859f;
const float PLANE_COLOR_B = 0.859f;
const float GRID_COLOR_R = 0.102f;
const float GRID_COLOR_G = 0.984f;
const float GRID_COLOR_B = 0.984f;
const float GRID_OFFSET = 0.001f; // 统一网格线偏移量
const float GRID_Y_OFFSET = 0.01f; // 新增：网格线y方向位移量

struct PlanePara
{
    QVector3D origin;
    QVector3D normal;
    QVector3D localX;
    QVector3D localY;
    float halfLength;
    unsigned int gridCntPerEdge;//网格线数量
    float offset;
};

// 继承 QOpenGLFunctions_3_3_Core 以绑定 OpenGL 函数
class WorkPlane : public QOpenGLFunctions_3_3_Core
{
public:
    WorkPlane()
        : m_planeVAO(0), m_planeVBO(0), m_planeEBO(0)
        , m_planeIndexCount(0)
        , m_gridVAO(0), m_gridVBO(0)
        , m_gridVertexCount(0)
    {
    }

    ~WorkPlane()
    {
        // 释放 OpenGL 资源
        glDeleteVertexArrays(1, &m_planeVAO);
        glDeleteBuffers(1, &m_planeVBO);
        glDeleteBuffers(1, &m_planeEBO);
        glDeleteVertexArrays(1, &m_gridVAO);
        glDeleteBuffers(1, &m_gridVBO);
    }

    void initPlane(const PlanePara& para)
    {
        // 初始化 OpenGL 函数（必须调用，绑定上下文）
        initializeOpenGLFunctions();

        // 复制传入参数到成员变量
        m_para = para;

        // 生成局部坐标系
        getLocalXY(m_para.normal, m_para.localX, m_para.localY);

        // 清空之前的网格点数据（避免重复初始化时数据叠加）
        m_gridPoints.clear();
        // 绘制平面和网格线
        generatePlaneBase(m_para);
        generateGridLines(m_para);
    }

    void drawPlane()
    {
        // 绘制平面基底
        glBindVertexArray(m_planeVAO);
        glDrawElements(GL_TRIANGLES, m_planeIndexCount, GL_UNSIGNED_INT, 0);

        // 绘制网格线
        glBindVertexArray(m_gridVAO);
        glDrawArrays(GL_LINES, 0, m_gridVertexCount);

        // 解绑 VAO
        glBindVertexArray(0);
    }

    // 计算射线与工作平面的交点（返回是否相交）
    bool getRayPlaneIntersection(
        const QVector3D& rayOrigin,    // 射线起点（摄像机位置）
        const QVector3D& rayDir,      // 射线方向
        const PlanePara& plane,       // 工作平面参数
        QVector3D& outIntersection    // 输出交点
    ) {
        // 平面方程：(P - origin) · normal = 0
        float denominator = QVector3D::dotProduct(rayDir, plane.normal);
        if (qAbs(denominator) < 1e-6) return false; // 射线与平面平行，无交点

        float t = QVector3D::dotProduct(plane.origin - rayOrigin, plane.normal) / denominator;
        if (t < 0) return false; // 交点在射线后方（摄像机背后）

        outIntersection = rayOrigin + rayDir * t;
        return true;
    }

    // 获取网格交叉点列表
    const std::vector<QVector3D>& gridPoints() const { return m_gridPoints; }
    // 获取平面参数（origin、normal等）
    const PlanePara& para() const { return m_para; }

private:
    void getLocalXY(const QVector3D& normal, QVector3D& localX, QVector3D& localY)
    {
        QVector3D ref = (qAbs(normal.y()) < 0.999f) ? QVector3D(0, 1, 0) : QVector3D(1, 0, 0);
        localX = QVector3D::crossProduct(normal, ref).normalized();
        localY = QVector3D::crossProduct(normal, localX).normalized();
    }

    void generatePlaneBase(const PlanePara& para)
    {
        //计算出平面四个顶点
        QVector3D pts[4];
        pts[0] = para.origin + (para.halfLength * -para.localX) + (para.halfLength * -para.localY) + (para.offset * para.normal);
        pts[1] = para.origin + (para.halfLength * para.localX) + (para.halfLength * -para.localY) + (para.offset * para.normal);
        pts[2] = para.origin + (para.halfLength * para.localX) + (para.halfLength * para.localY) + (para.offset * para.normal);
        pts[3] = para.origin + (para.halfLength * -para.localX) + (para.halfLength * para.localY) + (para.offset * para.normal);

        // 平面顶点数据（使用常量颜色）
        std::vector<float> vertices = {
            pts[0].x(), pts[0].y(), pts[0].z(), PLANE_COLOR_R, PLANE_COLOR_G, PLANE_COLOR_B,
            pts[1].x(), pts[1].y(), pts[1].z(), PLANE_COLOR_R, PLANE_COLOR_G, PLANE_COLOR_B,
            pts[2].x(), pts[2].y(), pts[2].z(), PLANE_COLOR_R, PLANE_COLOR_G, PLANE_COLOR_B,
            pts[3].x(), pts[3].y(), pts[3].z(), PLANE_COLOR_R, PLANE_COLOR_G, PLANE_COLOR_B
        };

        unsigned int indices[] = { 0, 1, 2, 0, 2, 3 };
        m_planeIndexCount = sizeof(indices) / sizeof(unsigned int);

        // 生成并配置 VAO/VBO/EBO
        glGenVertexArrays(1, &m_planeVAO);
        glGenBuffers(1, &m_planeVBO);
        glGenBuffers(1, &m_planeEBO);

        glBindVertexArray(m_planeVAO);
        glBindBuffer(GL_ARRAY_BUFFER, m_planeVBO);
        glBufferData(GL_ARRAY_BUFFER, vertices.size() * sizeof(float), vertices.data(), GL_STATIC_DRAW);

        glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, m_planeEBO);
        glBufferData(GL_ELEMENT_ARRAY_BUFFER, sizeof(indices), indices, GL_STATIC_DRAW);

        // 配置顶点位置属性
        glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 6 * sizeof(float), (void*)0);
        glEnableVertexAttribArray(0);
        // 配置顶点颜色属性
        glVertexAttribPointer(1, 3, GL_FLOAT, GL_FALSE, 6 * sizeof(float), (void*)(3 * sizeof(float)));
        glEnableVertexAttribArray(1);

        glBindVertexArray(0);
    }

    void generateGridLines(const PlanePara& para)
    {
        std::vector<float> gridVertices;
        float step = 2.0f * para.halfLength / para.gridCntPerEdge;

        // 计算网格交叉点（同样添加y方向位移）
        {
            std::vector<float> xCoords, yCoords;
            xCoords.push_back(-para.halfLength);
            yCoords.push_back(-para.halfLength);
            for (int i = 1; i < para.gridCntPerEdge; ++i) {
                xCoords.push_back(-para.halfLength + i * step);
                yCoords.push_back(-para.halfLength + i * step);
            }
            xCoords.push_back(para.halfLength);
            yCoords.push_back(para.halfLength);

            for (float x : xCoords) {
                for (float y : yCoords) {
                    QVector3D point = para.origin
                        + x * para.localX
                        + y * para.localY
                        + para.offset * para.normal;
                    point.setY(point.y() + GRID_Y_OFFSET); // 网格交叉点y方向位移
                    m_gridPoints.push_back(point);
                }
            }
        }

        // X方向网格线（平行于localX）
        for (int i = 1; i < para.gridCntPerEdge; ++i)
        {
            float yPos = -para.halfLength + i * step;

            QVector3D start = para.origin - para.halfLength * para.localX + yPos * para.localY;
            QVector3D end = para.origin + para.halfLength * para.localX + yPos * para.localY;

            start += para.offset * para.normal;
            end += para.offset * para.normal;

            // 关键修改1：X方向网格线y位移
            start.setY(start.y() + GRID_Y_OFFSET);
            end.setY(end.y() + GRID_Y_OFFSET);

            gridVertices.insert(gridVertices.end(),
                {
                    start.x(), start.y(), start.z(), GRID_COLOR_R, GRID_COLOR_G, GRID_COLOR_B,
                    end.x(),   end.y(),   end.z(),   GRID_COLOR_R, GRID_COLOR_G, GRID_COLOR_B
                });
        }

        // Y方向网格线（平行于localY）
        for (int i = 1; i < para.gridCntPerEdge; ++i)
        {
            float xPos = -para.halfLength + i * step;

            QVector3D start = para.origin - para.halfLength * para.localY + xPos * para.localX;
            QVector3D end = para.origin + para.halfLength * para.localY + xPos * para.localX;

            start += para.offset * para.normal;
            end += para.offset * para.normal;

            // 关键修改2：Y方向网格线y位移
            start.setY(start.y() + GRID_Y_OFFSET);
            end.setY(end.y() + GRID_Y_OFFSET);

            gridVertices.insert(gridVertices.end(),
                {
                    start.x(), start.y(), start.z(), GRID_COLOR_R, GRID_COLOR_G, GRID_COLOR_B,
                    end.x(),   end.y(),   end.z(),   GRID_COLOR_R, GRID_COLOR_G, GRID_COLOR_B
                });
        }

        m_gridVertexCount = gridVertices.size() / 6;

        // 生成并配置 VAO/VBO
        glGenVertexArrays(1, &m_gridVAO);
        glGenBuffers(1, &m_gridVBO);

        glBindVertexArray(m_gridVAO);
        glBindBuffer(GL_ARRAY_BUFFER, m_gridVBO);
        glBufferData(GL_ARRAY_BUFFER, gridVertices.size() * sizeof(float), gridVertices.data(), GL_STATIC_DRAW);

        // 配置顶点位置属性
        glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 6 * sizeof(float), (void*)0);
        glEnableVertexAttribArray(0);
        // 配置顶点颜色属性
        glVertexAttribPointer(1, 3, GL_FLOAT, GL_FALSE, 6 * sizeof(float), (void*)(3 * sizeof(float)));
        glEnableVertexAttribArray(1);

        glBindVertexArray(0);
    }

private:
    PlanePara m_para;//平面参数
    unsigned int m_planeVAO, m_planeVBO, m_planeEBO;
    int m_planeIndexCount;
    unsigned int m_gridVAO, m_gridVBO;
    int m_gridVertexCount;
    std::vector<QVector3D> m_gridPoints; // 存储所有网格交叉点
};