#pragma once
#include <qvector3d.h>

struct Vector3f {
	float x;  // x分量
	float y;  // y分量
	float z;  // z分量

	// 构造函数：初始化 x, y, z
	Vector3f(float x_ = 0.0f, float y_ = 0.0f, float z_ = 0.0f)
		: x(x_), y(y_), z(z_) {}

	// 从 QVector3D 转换（代码中用 QVector3D 构造 Vector3f）
	Vector3f(const QVector3D& qv)
		: x(qv.x()), y(qv.y()), z(qv.z()) {}

	// 向量点积（可能用于角度计算）
	float dot(const Vector3f& other) const {
		return x * other.x + y * other.y + z * other.z;
	}

	// 向量叉积（可能用于计算垂直向量）
	Vector3f cross(const Vector3f& other) const {
		return Vector3f(
			y * other.z - z * other.y,
			z * other.x - x * other.z,
			x * other.y - y * other.x
		);
	}

	// 向量归一化（单位向量）
	Vector3f normalized() const {
		float len = sqrt(x*x + y*y + z*z);
		if (len < 1e-9) return Vector3f();
		return Vector3f(x / len, y / len, z / len);
	}

	// 重载：向量 - 向量（两个向量的分量相减）
	Vector3f operator-(const Vector3f& other) const {
		return Vector3f(
			x - other.x,   // x分量相减
			y - other.y,   // y分量相减
			z - other.z    // z分量相减
		);
	}

	// 重载：一元负运算符（向量取反）
	Vector3f operator-() const {
		return Vector3f(
			-x,  // x分量取反
			-y,  // y分量取反
			-z   // z分量取反
		);
	}

	// 重载：向量 × 标量（返回新向量）
	Vector3f operator*(float scalar) const {
		return Vector3f(
			x * scalar,
			y * scalar,
			z * scalar
		);
	}

	// 可选：标量 × 向量（为了支持 scalar * vector 的写法）
	friend Vector3f operator*(float scalar, const Vector3f& vec) {
		return Vector3f(
			vec.x * scalar,
			vec.y * scalar,
			vec.z * scalar
		);
	}

	// 计算平面上两个向量的夹角（代码中用到的 AngleOnPlaneTo）
	// 参数：other 目标向量；planeNormal 平面法向量（用于确定旋转平面）
	float AngleOnPlaneTo(const Vector3f& other, const Vector3f& planeNormal) const {
		// 1. 将当前向量和目标向量投影到平面上（消除法向量方向分量）
		// 修正步骤：1. 确保平面法向量是单位向量；2. 计算向量在法向量上的投影；3. 减去投影得到平面内分量
		Vector3f normalizedPlaneNormal = planeNormal.normalized(); // 法向量归一化（关键）

																   // 向量v1在平面上的投影 = 原向量 - （原向量在法向量上的投影长度 × 单位法向量）
		float projLen1 = this->dot(normalizedPlaneNormal); // 投影长度（标量）
		Vector3f v1 = *this - normalizedPlaneNormal * projLen1;

		// 向量v2在平面上的投影（同理）
		float projLen2 = other.dot(normalizedPlaneNormal);
		Vector3f v2 = other - normalizedPlaneNormal * projLen2;

		// 后续：归一化平面内向量，避免长度影响夹角计算
		v1 = v1.normalized();
		v2 = v2.normalized();

		// 2. 计算平面内的夹角（用点积求余弦值，再转角度）
		float dotProd = v1.dot(v2);
		dotProd = qBound(-1.0f, dotProd, 1.0f);  // 避免精度问题导致的NaN
		float angleRad = acos(dotProd);

		// 3. 可能通过叉积判断方向（顺/逆时针），返回带符号角度
		Vector3f crossProd = v1.cross(v2);
		if (crossProd.dot(planeNormal) < 0) {
			angleRad = -angleRad;  // 方向相反时角度取负
		}
		return angleRad;
	}

	// 计算当前向量与目标向量 dir 之间的夹角（单位：弧度，范围 0~π）
	float Angle(const Vector3f& dir) const {
		// 1. 先将两个向量归一化（避免长度影响夹角计算）
		Vector3f vec1 = this->normalized();
		Vector3f vec2 = dir.normalized();

		// 2. 计算点积（点积 = |vec1|×|vec2|×cosθ，归一化后 |vec1|=|vec2|=1，所以点积=cosθ）
		float dotProduct = vec1.dot(vec2);

		// 3. 限制点积范围在 [-1.0f, 1.0f]（避免浮点精度误差导致 acos 返回 NaN）
		dotProduct = qBound(-1.0f, dotProduct, 1.0f); // 依赖 <QtGlobal>，若不依赖可手动判断

													  // 4. 通过反余弦函数计算夹角（弧度），范围 0~π
		return acos(dotProduct);
	}
};