#pragma once
#include <QOpenGLFunctions_4_5_Core>
#include <QOpenGLShaderProgram>
#include <QVector3D>
#include <vector>
#include <qdebug.h>

// 定义全局无效点常量（三个分量均为正无穷大）
const QVector3D INVALID_POINT(
    std::numeric_limits<float>::infinity(),  // x分量：正无穷大
    std::numeric_limits<float>::infinity(),  // y分量：正无穷大
    std::numeric_limits<float>::infinity()   // z分量：正无穷大
);

class LineDrawer : protected QOpenGLFunctions_4_5_Core
{
public:
    LineDrawer() {
        m_isDrawing = false;
        m_selectedIndex = -1; // 初始无选中线段
        m_selectedColor = QVector4D(1.0f, 0.0f, 0.0f, 1.0f); // 选中颜色（红色）
    }
    ~LineDrawer() { cleanup(); }

    // 初始化VAO/VBO（在initializeGL中调用）
    void init()
    {
        initializeOpenGLFunctions();
        // 创建VAO和VBO
        glGenVertexArrays(1, &m_lineVAO);
        glGenBuffers(1, &m_lineVBO);
        // 初始顶点数据（起点+终点，默认原点）
        m_vertices = { 0.0f, 0.0f, 0.0f, 0.0f, 0.0f, 0.0f };
    }

    // 获取所有端点的接口
    std::vector<QVector3D> getAllEndpoints() const {
        std::vector<QVector3D> points;
        for (const auto& line : m_savedLines) {
            points.push_back(line.first);
            points.push_back(line.second);
        }
        return points;
    }

    // 获取所有线段（用于点击检测）
    const std::vector<std::pair<QVector3D, QVector3D>>& getAllLines() const {
        return m_savedLines;
    }

    // 设置直线颜色（支持当前绘制线、已保存线和选中线）
    void setCurrentLineColor(const QVector4D& color) { m_currentLineColor = color; }
    void setSavedLineColor(const QVector4D& color) { m_savedLineColor = color; }
    void setSelectedColor(const QVector4D& color) { m_selectedColor = color; }

    // 删除所有直线
    void clearAllLines() {
        m_isDrawing = false;
        m_savedLines.clear();
        m_selectedIndex = -1; // 清空选中状态
        m_vertices = { 0.0f, 0.0f, 0.0f, 0.0f, 0.0f, 0.0f };
        updateVertexData();
    }

    // 撤销最近绘制的线段
    void undoLastLine() {
        if (!m_savedLines.empty()) {
            m_savedLines.pop_back();
            // 若撤销的是选中线段，取消选中状态
            if (m_selectedIndex == m_savedLines.size()) {
                m_selectedIndex = -1;
            }
        }
    }

    //删除选中的直线段
    void deleteSelectedLine()
    {
        if (!m_savedLines.empty() && (m_selectedIndex < m_savedLines.size()))
        {
            m_savedLines.erase(m_savedLines.begin() + m_selectedIndex);
            m_delRecordStartPos = m_savedLines[m_selectedIndex].first;
            m_selectedIndex = -1;
        }
    }

    // 开始绘制
    void startDraw(const QVector3D& startPos)
    {
        m_isDrawing = true;
        m_startPos = startPos;
        m_endPos = startPos;
        updateVertexData();
    }

    // 更新绘制
    void updateDraw(const QVector3D& currentPos)
    {
        if (!m_isDrawing) return;
        m_endPos = currentPos;
        updateVertexData();
    }

    //保存已经绘制的直线
    void saveDrawnLine()
    {
        m_isDrawing = false;
        m_savedLines.push_back({ m_startPos, m_endPos });
    }

    // 结束绘制
    void finishDraw()
    {
        m_isDrawing = false;
        m_selectedIndex = -1;//恢复未选中状态
        if (m_delRecordStartPos != m_startPos)
        {
            m_savedLines.push_back({ m_startPos, m_endPos });
        }

        m_delRecordStartPos = INVALID_POINT;
    }

    void draw(QOpenGLShaderProgram& shader)
    {
        if (!m_isDrawing && m_savedLines.empty()) return;

        shader.bind();
        glBindVertexArray(m_lineVAO);
        glLineWidth(2.0f); // 线宽（可调整）

        // 1. 绘制当前正在绘制的直线（保持原逻辑）
        if (m_isDrawing)
        {
            shader.setUniformValue("lineColor", m_currentLineColor);
            glDrawArrays(GL_LINES, 0, 2);
        }

        // 2. 绘制已保存的线段（关键修改：逐线段绘制，区分选中状态）
        if (!m_savedLines.empty())
        {
            // 临时存储单条线段的顶点数据（2个点，每个点3个坐标）
            std::vector<float> singleLineVertices(6);

            for (size_t i = 0; i < m_savedLines.size(); ++i)
            {
                const auto& line = m_savedLines[i];
                // 填充当前线段的顶点数据
                singleLineVertices[0] = line.first.x();  // 起点X
                singleLineVertices[1] = line.first.y();  // 起点Y
                singleLineVertices[2] = line.first.z();  // 起点Z
                singleLineVertices[3] = line.second.x(); // 终点X
                singleLineVertices[4] = line.second.y(); // 终点Y
                singleLineVertices[5] = line.second.z(); // 终点Z

                // 3. 设置当前线段的颜色（选中则红色，否则默认色）
                if (i == m_selectedIndex)
                {
                    // 选中颜色：红色（RGBA）
                    shader.setUniformValue("lineColor", m_selectedColor);
                }
                else
                {
                    // 未选中：默认已保存线颜色
                    shader.setUniformValue("lineColor", m_savedLineColor);
                }

                // 4. 更新VBO并绘制当前线段
                glBindBuffer(GL_ARRAY_BUFFER, m_lineVBO);
                glBufferData(GL_ARRAY_BUFFER, singleLineVertices.size() * sizeof(float),
                singleLineVertices.data(), GL_DYNAMIC_DRAW); // 动态更新单条线段
                // 设置顶点属性（位置）
                glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 3 * sizeof(float), (void*)0);
                glEnableVertexAttribArray(0);
                // 绘制单条线段（GL_LINES 模式，2个顶点）
                glDrawArrays(GL_LINES, 0, 2);
            }
        }

        glBindVertexArray(0);
        shader.release();
    }

    // 清理资源
    void cleanup()
    {
        glDeleteVertexArrays(1, &m_lineVAO);
        glDeleteBuffers(1, &m_lineVBO);
        m_vertices.clear();
        m_savedLines.clear();
        m_selectedIndex = -1;
    }

    // 选中状态管理
    void setSelectedIndex(int index) {
        // 索引有效性检查
        if (index < -1 || static_cast<size_t>(index) >= m_savedLines.size()) {
            m_selectedIndex = -1;
        }
        else {
            m_selectedIndex = index;
        }
    }

    int getSelectedIndex() const { return m_selectedIndex; }

    QVector3D getCurrentLineStart() const {
        return m_startPos;
    }

    bool isDrawing() const { return m_isDrawing; }

private:
    // 更新顶点数据到VBO
    void updateVertexData()
    {
        m_vertices[0] = m_startPos.x();
        m_vertices[1] = m_startPos.y();
        m_vertices[2] = m_startPos.z();
        m_vertices[3] = m_endPos.x();
        m_vertices[4] = m_endPos.y();
        m_vertices[5] = m_endPos.z();

        glBindVertexArray(m_lineVAO);
        glBindBuffer(GL_ARRAY_BUFFER, m_lineVBO);
        glBufferData(GL_ARRAY_BUFFER, m_vertices.size() * sizeof(float), m_vertices.data(), GL_DYNAMIC_DRAW);
        glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 3 * sizeof(float), (void*)0);
        glEnableVertexAttribArray(0);
        glBindVertexArray(0);
    }

private:
    unsigned int m_lineVAO = 0, m_lineVBO = 0;
    std::vector<float> m_vertices; // 顶点数据：[x1,y1,z1, x2,y2,z2]
    QVector3D m_startPos, m_endPos;
    QVector3D m_delRecordStartPos = INVALID_POINT;//记录已经删除的直线的起点
    bool m_isDrawing;//正在绘制（鼠标左键按下置为true, 鼠标移动时都为true, 左键松开置为false）
    std::vector<std::pair<QVector3D, QVector3D>> m_savedLines; // 保存已完成线段
    int m_selectedIndex; // 当前选中线段索引（-1表示无选中）

    // 颜色属性
    float m_fixedDepth = -5.0f;
    QVector4D m_currentLineColor = { 1.0f, 0.0f, 0.0f, 1.0f }; // 当前绘制线（红色）
    QVector4D m_savedLineColor = { 0.0f, 1.0f, 0.0f, 1.0f };   // 已保存线（绿色）
    QVector4D m_selectedColor = { 1.0f, 1.0f, 0.0f, 1.0f };   // 点击选中的线（绿色）
};