#include <qopenglwidget.h>
#include <qopenglfunctions_4_5_core.h>
#include <qopenglshaderprogram.h>
#include <qopenglvertexarrayobject.h>
#include <qopenglbuffer.h>
#include <QMouseEvent>
#include "camera.h"
#include "workPlane.h"

class GLWidget : public QOpenGLWidget ,protected QOpenGLFunctions_4_5_Core
{
	Q_OBJECT

public:
	GLWidget(QWidget* parent = nullptr)
	{
		setMouseTracking(true);
	}
	~GLWidget() {}

	void initializeGL() override
	{
		initializeOpenGLFunctions();

		initShader(m_shader, QString("./shader/LoadingShader.vert"), QString("./shader/LoadingShader.frag"));
		initTriangle();
		initPlane();
	}

	void resizeGL(int width,int height) override
	{
		glViewport(0, 0, width, height);
		m_camera.Init(width, height);

		m_projectionMat.setToIdentity();
		m_projectionMat.perspective(45.0f, width / height, 0.1f, 100.0f);
	}

	void paintGL() override
	{
		glEnable(GL_DEPTH_TEST);

		m_shader.bind();
		
		m_modelMat.setToIdentity();

		m_viewMat = m_camera.getViewMatrix();

		//m_viewMat.lookAt(QVector3D(0.0f, 0.0f, 5.0f), QVector3D(0.0f, 0.0f, 0.0f), QVector3D(0.0f, 1.0f, 0.0f));
		m_shader.setUniformValue("model", m_modelMat);
		m_shader.setUniformValue("view", m_viewMat);
		m_shader.setUniformValue("projection", m_projectionMat);
		drawTriangle();

		m_workPlane.drawPlane();
		m_shader.release();
	}

	void initShader(QOpenGLShaderProgram& shader,const QString& vertShaderPath,const QString& fragShaderPath)
	{
		if (!shader.addShaderFromSourceFile(QOpenGLShader::Vertex, vertShaderPath) || 
			!shader.addShaderFromSourceFile(QOpenGLShader::Fragment, fragShaderPath) || 
			!shader.link())
		{
			qDebug() << shader.log();
		}
	}

	void initPlane()
	{
		PlanePara para;
		para.origin = QVector3D(0.0f, -1.0f, 0.0f);
		para.normal = QVector3D(0.0f, 1.0f, 0.0f); // ϵķYᣩ
		para.halfLength = 25.0f;
		para.gridCntPerEdge = 32;
		para.offset = 0.0f; // ƽƫƣY=0.1f

		m_workPlane.initPlane(para);
	}

	void initTriangle()
	{
		float vertices[] =
		{
			//position       color 
			-0.5f,0.0f,0.5f, 0.5f,0.5f,0.0f,
			0.0f,0.5f,0.5f, 0.5f,0.5f,0.5f,
			0.5f,0.0f,0.5f, 0.5f,0.0f,0.5f,
		};

		glGenVertexArrays(1, &m_vao);
		glGenBuffers(1, &m_vbo);

		glBindVertexArray(m_vao);
		glBindBuffer(GL_ARRAY_BUFFER, m_vbo);

		glBufferData(GL_ARRAY_BUFFER, sizeof(vertices), vertices, GL_STATIC_DRAW);
		glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 6 * sizeof(float), (void*)0);
		glEnableVertexAttribArray(0);
		glVertexAttribPointer(1, 3, GL_FLOAT, GL_FALSE, 6 * sizeof(float), (void*)(3 * sizeof(float)));
		glEnableVertexAttribArray(1);

		glBindVertexArray(0);
	}

	void drawTriangle()
	{
		glBindVertexArray(m_vao);
		glDrawArrays(GL_TRIANGLES, 0, 3);
		glBindVertexArray(0);
	}

protected:
	bool event(QEvent* e) override
	{
		m_camera.handle(e);
		update();

		return QWidget::event(e);
	}

private:
	QOpenGLShaderProgram m_shader;
	unsigned int m_vao, m_vbo;

	QMatrix4x4 m_modelMat;
	QMatrix4x4 m_viewMat;
	QMatrix4x4 m_projectionMat;
	Camera m_camera = Camera(QVector3D(0.0f, 0.0f, 5.0f));
	WorkPlane m_workPlane;
};